#include "Common.h"
#include "ScopeSemaphoreExit.h"

Common::Common(){}

QString Common::getTimeString() {
    QDateTime currentDateTime = QDateTime::currentDateTime();
    QString formattedDateTime = currentDateTime.toString("yyyy-MM-dd hh:mm:ss");
    return formattedDateTime;
}
QString Common::timestampToDateString(qint64 timestamp) {
    QDateTime dateTime = QDateTime::fromSecsSinceEpoch(timestamp);
    QString formattedDate = dateTime.toString("yyyy-MM-dd hh:mm:ss");
    return formattedDate;
}
QString Common::generateSignature(const QString& accessKeySecret, const QString& verb,
                                  const QString& contentMD5, const QString& contentType,
                                  const QString& date, const QString& ossHeaders,
                                  const QString& canonicalizedResource)
{
    // 构造规范化的字符串
    QString stringToSign = verb + "\n" +
            contentMD5 + "\n" +
            contentType + "\n" +
            date + "\n" +
            ossHeaders+ "\n" +  // 添加'\n'分隔符，并确保ossHeaders末尾没有多余的空白
            canonicalizedResource;
    
    // 将密钥和消息转换为字节数组
    QByteArray hmacKey = accessKeySecret.toUtf8();
    QByteArray message = stringToSign.toUtf8();
    // 使用HMAC-SHA1计算签名
    QMessageAuthenticationCode mac(QCryptographicHash::Sha1);
    mac.setKey(hmacKey);
    mac.addData(message);
    
    
    QByteArray signature = mac.result().toBase64();  // 直接使用QMessageAuthenticationCode的结果
    
    return QString(signature);
}
QString Common::getVideoOut(){
    return videoOut;
}
void Common::setVideoOut(QString videoOut){
    videoOut.append("/");
    this->videoOut=videoOut;
}


QString Common::getVideoDownload(){
    return videoDownload;
}
void Common::setVideoDownload(QString videoDownload){
    videoDownload.append("/");
    this->videoDownload=videoDownload;
}

QString Common::getImages(){
    return images;
}
void Common::setImages(QString images){
    images.append("/");
    this->images=images;
}

QString Common::DecIpToHexIp(const QString& decIp) {
    // 将 IP 地址按点号分割成多个部分
    QStringList parts = decIp.split(".");

    // 检查 IP 地址是否有效（应包含 4 个部分）
    if (parts.size() != 4) {
        return QString(); // 如果无效，返回空字符串
    }

    QString hexIp; // 用于存储最终的十六进制表示

    // 从最后一个部分开始遍历
    for (int i = 3; i >= 0; --i) {
        // 将每个部分转换为整数
        bool ok;
        int part = parts[i].toInt(&ok);
        if (!ok) return QString(); // 如果转换失败，返回空字符串

        // 将整数转换为十六进制字符串，并在必要时补零
        QString hexPart = QString::number(part, 16).rightJustified(2, '0');

        // 将十六进制字符串添加到结果中
        hexIp.append(hexPart);
    }

    // 在结果前添加 "0x" 以表示十六进制
    hexIp.prepend("0x");

    return hexIp; // 返回最终的十六进制表示
}
QString Common::GetLocalIp() {
    QString ipAddress;
    QList<QHostAddress> list = QNetworkInterface::allAddresses();
    for (const QHostAddress& address : list) {
        if (address != QHostAddress::LocalHost && address.toIPv4Address()) {
            ipAddress = address.toString();
            break;
        }
    }
    if (ipAddress.isEmpty()) {
        ipAddress = QHostAddress(QHostAddress::LocalHost).toString();
    }
    return ipAddress;
}

// 计算校验和
uint16_t Common::calculateChecksum(uint16_t *buffer, int size) {
    uint32_t checksum = 0;
    while (size > 1) {
        checksum += *buffer++;
        size -= sizeof(uint16_t);
    }
    if (size) {
        checksum += *(uint8_t *)buffer;
    }
    checksum = (checksum >> 16) + (checksum & 0xffff);
    checksum += (checksum >> 16);
    return (uint16_t)(~checksum);
}

// 发送 ICMP 请求
bool Common::sendPingRequest(int sockfd, const char *ipAddress, int sequence) {
    struct sockaddr_in destAddr;
    memset(&destAddr, 0, sizeof(destAddr));
    destAddr.sin_family = AF_INET;
    inet_pton(AF_INET, ipAddress, &destAddr.sin_addr);

    struct  ICMPHeader icmpHeader;
    icmpHeader.type = ICMP_ECHO;
    icmpHeader.code = 0;
    icmpHeader.id = getpid() & 0xffff;
    icmpHeader.sequence = sequence;
    icmpHeader.checksum = 0;

    char packet[sizeof(struct ICMPHeader) + 32];
    memcpy(packet, &icmpHeader, sizeof(icmpHeader));
    memset(packet + sizeof(icmpHeader), 0, 32);

    icmpHeader.checksum = calculateChecksum((uint16_t *)packet, sizeof(packet));
    memcpy(packet, &icmpHeader, sizeof(icmpHeader));

    if (sendto(sockfd, packet, sizeof(packet), 0, (struct sockaddr *)&destAddr, sizeof(destAddr)) < 0) {
        perror("sendto");
        return false;
    }
    return true;
}

// 接收 ICMP 回复
bool Common::receivePingReply(int sockfd, int sequence) {
    char buffer[1024];
    struct sockaddr_in srcAddr;
    socklen_t srcAddrLen = sizeof(srcAddr);

    ssize_t bytesRead = recvfrom(sockfd, buffer, sizeof(buffer), 0, (struct sockaddr *)&srcAddr, &srcAddrLen);
    if (bytesRead < 0) {
        perror("recvfrom");
        return false;
    }

    struct iphdr *ipHeader = (struct iphdr *)buffer;
    struct ICMPHeader *icmpHeader = (struct ICMPHeader *)(buffer + (ipHeader->ihl << 2));

    if (icmpHeader->type == ICMP_ECHOREPLY && icmpHeader->sequence == sequence) {
        std::cout << "Ping successful!" << std::endl;
        return true;
    }

    return false;
}
bool Common::pingAddress(const QString &address) {

    QByteArray && byJsonIp =address.toUtf8();
    const char * ipAddress= byJsonIp.data();

    int sockfd = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
    ScopeSemaphoreExit guard([sockfd]() {
        close(sockfd);
    });
    if (sockfd < 0) {
        perror("socket");
        return 1;
    }

    int sequence = 1;
    if (sendPingRequest(sockfd, ipAddress, sequence)) {
        if (receivePingReply(sockfd, sequence)) {
            qInfo()<<"Ping to " << ipAddress << " successful!";
            return true;
        } else {
            qInfo()<<"Ping to " << ipAddress << " failed!";
            return false;
        }
    }
    return false;
}
void Common::get_network_info(const char *interface, QString &mac, QString &subnet_mask, QString &gateway) {
    int fd;
    struct ifreq ifr;
    struct rtentry route;
    unsigned char *mac_addr;
    struct sockaddr_in *addr;
    struct ifaddrs *ifaddr, *ifa;

    // Create a socket to perform ioctl requests
    fd = socket(AF_INET, SOCK_DGRAM, 0);
    if (fd < 0) {
        perror("socket");
        return;
    }

    // Get the MAC address
    memset(&ifr, 0, sizeof(ifr));
    strncpy(ifr.ifr_name, interface, IFNAMSIZ - 1);

    if (ioctl(fd, SIOCGIFHWADDR, &ifr) == 0) {
        mac_addr = (unsigned char *)ifr.ifr_hwaddr.sa_data;
        mac = QString::asprintf("%02x:%02x:%02x:%02x:%02x:%02x",
            mac_addr[0], mac_addr[1], mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5]);
    } else {
        perror("SIOCGIFHWADDR");
        mac = "Unknown";
    }

    // Get the subnet mask
    if (ioctl(fd, SIOCGIFNETMASK, &ifr) == 0) {
        addr = (struct sockaddr_in *)&ifr.ifr_netmask;
        subnet_mask = QString::fromUtf8(inet_ntoa(addr->sin_addr));
    } else {
        perror("SIOCGIFNETMASK");
        subnet_mask = "Unknown";
    }

    // Get the default gateway
    FILE *fp = fopen("/proc/net/route", "r");
    if (fp != NULL) {
        char buffer[256];
        while (fgets(buffer, sizeof(buffer), fp)) {
            char iface[IFNAMSIZ];
            unsigned long dest, gateway_addr;
            if (sscanf(buffer, "%s %lx %lx", iface, &dest, &gateway_addr) == 3) {
                if (dest == 0) { // Default route
                    struct in_addr gwaddr;
                    gwaddr.s_addr = gateway_addr;
                    gateway = QString::fromUtf8(inet_ntoa(gwaddr));
                    break;
                }
            }
        }
        fclose(fp);
    } else {
        perror("fopen");
        gateway = "Unknown";
    }

    // Clean up
    close(fd);
}

// 确定当前网络接口
void Common::determine_interface(char *interface) {
    struct ifaddrs *ifaddr, *ifa;
    int family, s;
    char host[NI_MAXHOST];

    if (getifaddrs(&ifaddr) == -1) {
        perror("getifaddrs");
        exit(EXIT_FAILURE);
    }

    // Walk through linked list, maintaining head pointer so we can free list later
    for (ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next) {
        if (ifa->ifa_addr == NULL)
            continue;

        family = ifa->ifa_addr->sa_family;

        // Check for IPv4 or IPv6
        if (family == AF_INET || family == AF_INET6) {
            s = getnameinfo(ifa->ifa_addr,
                            (family == AF_INET) ? sizeof(struct sockaddr_in) : sizeof(struct sockaddr_in6),
                            host, NI_MAXHOST, NULL, 0, NI_NUMERICHOST);
            if (s != 0) {
                printf("getnameinfo() failed: %s\n", gai_strerror(s));
                exit(EXIT_FAILURE);
            }

            // Check if the interface is up and running
            if (ifa->ifa_flags & IFF_UP && ifa->ifa_flags & IFF_RUNNING) {
                strncpy(interface, ifa->ifa_name, IFNAMSIZ - 1);
                break;
            }
        }
    }

    freeifaddrs(ifaddr);
}
Common::~Common(){}
